/* @rto/sdk v7.10.1 */
export type RtoAction =
  | "detect"
  | "open"
  | "focus"
  | "navigate"
  | "close"
  | "domClick"
  | "domType"
  | "submit"
  | "highlight"
  | "runJs"
  | "command"
  | "listTabs";

export type RtoErrorCode =
  | "DOMAIN_NOT_ALLOWED"
  | "NO_CONTROLLED_TAB"
  | "ELEMENT_NOT_FOUND"
  | "TIMEOUT"
  | "EXT_NOT_DETECTED";

export interface RtoTabStatusPush {
  origin: "plugin";
  type: "tabStatus";
  isOpen: boolean;
  tabKey?: string;
  tabId?: number;
  url?: string;
  title?: string;
}

export interface RtoClientOptions {
  timeoutMs?: number;
  eventFilter?: (ev: MessageEvent<any>) => boolean;
  target?: Window;
  requireDetect?: boolean;
}

function uid(): string {
  if (typeof crypto !== "undefined" && "randomUUID" in crypto) {
    return (crypto as any).randomUUID();
  }
  return `${Date.now()}_${Math.random().toString(16).slice(2)}`;
}

export class RtoClient {
  private opts: Required<RtoClientOptions>;
  private pending = new Map<string, { resolve: (v:any)=>void; reject:(e:any)=>void; timer:any; type:string }>();
  private listeners: { tabStatus: Set<(p:RtoTabStatusPush)=>void> } = {
    tabStatus: new Set()
  };

  constructor(opts: RtoClientOptions = {}) {
    this.opts = {
      timeoutMs: opts.timeoutMs ?? 12000,
      eventFilter: opts.eventFilter ?? (() => true),
      target: opts.target ?? window,
      requireDetect: opts.requireDetect ?? true
    };
    this.onMessage = this.onMessage.bind(this);
    window.addEventListener("message", this.onMessage);
  }

  dispose() {
    window.removeEventListener("message", this.onMessage);
    for (const [id, p] of this.pending) {
      clearTimeout(p.timer);
      p.reject(new Error("RTO client disposed"));
    }
    this.pending.clear();
    this.listeners.tabStatus.clear();
  }

  onTabStatus(cb: (push:RtoTabStatusPush)=>void) {
    this.listeners.tabStatus.add(cb);
    return () => this.listeners.tabStatus.delete(cb);
  }

  private onMessage(ev: MessageEvent<any>) {
    if (!this.opts.eventFilter(ev)) return;
    const msg = ev.data;
    if (!msg || typeof msg !== "object") return;

    // Push notifications
    if (msg.origin === "plugin" && msg.type === "tabStatus") {
      this.listeners.tabStatus.forEach(fn => {
        try { fn(msg as RtoTabStatusPush); } catch {}
      });
      return;
    }

    // Responses to pending requests
    const requestId = msg.requestId;
    if (requestId && this.pending.has(requestId)) {
      const pending = this.pending.get(requestId)!;
      if (msg.type === "RTO_RESPONSE") {
        clearTimeout(pending.timer);
        this.pending.delete(requestId);
        if (msg.ok === false || msg.error) {
          const code = msg.error?.code ?? "TIMEOUT";
          const err = new Error(msg.error?.message || `RTO error for ${pending.type}: ${code}`);
          (err as any).code = code;
          pending.reject(err);
        } else {
          pending.resolve(msg.data);
        }
      } else if (msg.type === "getHtmlResult") {
        clearTimeout(pending.timer);
        this.pending.delete(requestId);
        if (msg.status === "error") {
          const err = new Error(`RTO getHtml error for ${pending.type}`);
          (err as any).code = "ELEMENT_NOT_FOUND";
          pending.reject(err);
        } else {
          pending.resolve(msg.data);
        }
      }
    }
  }

  private send<T = any>(payload: any): Promise<T> {
    const requestId = payload.requestId ?? uid();
    const type = payload.type;
    const envelope = { ...payload, requestId };
    return new Promise<T>((resolve, reject) => {
      const timer = setTimeout(() => {
        this.pending.delete(requestId);
        const err = new Error(`RTO timeout for ${type}`);
        (err as any).code = "TIMEOUT";
        reject(err);
      }, this.opts.timeoutMs);
      this.pending.set(requestId, { resolve, reject, timer, type });
      this.opts.target.postMessage(envelope, "*");
    });
  }

  async detect(require = this.opts.requireDetect): Promise<{ ready: boolean; version?: string } | null> {
    try {
      const data = await this.send<{ ready:boolean; version?:string}>({
        origin: "admin-page",
        type: "RTO_REQUEST",
        action: "detect",
      });
      if (require && !(data && (data as any).ready)) {
        const err = new Error("RTO not detected");
        (err as any).code = "EXT_NOT_DETECTED";
        throw err;
      }
      return (data as any) ?? null;
    } catch (e:any) {
      if (!require) return null;
      throw e;
    }
  }

  open(params: { url: string; tabKey: string; focus?: boolean; newTab?: boolean; singleton?: boolean; }) {
    return this.send<void>({ origin: "admin-page", type: "open", ...params });
  }

  focus(tabKey: string) {
    return this.send<void>({ origin: "admin-page", type: "focus", tabKey });
  }

  navigate(params: { tabKey: string; url: string; focus?: boolean; }) {
    return this.send<void>({ origin: "admin-page", type: "navigate", ...params });
  }

  close(tabKey: string) {
    return this.send<void>({ origin: "admin-page", type: "close", tabKey });
  }

  domClick(params: { tabKey: string; selector: string; }) {
    return this.send<void>({ origin: "admin-page", type: "domClick", ...params });
  }

  domType(params: { tabKey: string; selector: string; text: string; focus?: boolean; select?: boolean; replace?: boolean; }) {
    return this.send<void>({ origin: "admin-page", type: "domType", ...params });
  }

  submit(params: { tabKey: string; selector: string; }) {
    return this.send<void>({ origin: "admin-page", type: "submit", ...params });
  }

  highlight(params: { tabKey: string; selector: string; color?: string; ms?: number; }) {
    return this.send<void>({ origin: "admin-page", type: "highlight", ...params });
  }

  runJs(params: { tabKey: string; code: string; }) {
    return self?.Promise ? this.send<any>({ origin: "admin-page", type: "runJs", ...params }) : this.send<any>({ origin: "admin-page", type: "runJs", ...params });
  }

  getHtml(params: { tabKey: string; selector: string; property?: "outerHTML" | "textContent" }) {
    return this.send<string>({
      origin: "admin-page",
      type: "command",
      action: "getHtml",
      tabKey: params.tabKey,
      payload: { selector: params.selector, property: params.property ?? "outerHTML" },
    });
  }

  listTabs() {
    return this.send<Array<{ tabKey:string; url?:string; title?:string }>>({
      origin: "admin-page",
      type: "listTabs"
    });
  }
}

export function createRtoClient(opts: RtoClientOptions = {}) {
  return new RtoClient(opts);
}
