import { describe, it, expect, beforeEach, afterEach } from 'vitest';
import { createRtoClient } from '../src/index';

function mockWindow() {
  const listeners = new Set<(e:MessageEvent)=>void>();
  const w: any = {
    _listeners: listeners,
    addEventListener: (t:string, cb:any) => { if (t==='message') listeners.add(cb); },
    removeEventListener: (t:string, cb:any) => { if (t==='message') listeners.delete(cb); },
    postMessage: (payload:any) => {
      // respond async like the extension would
      queueMicrotask(() => {
        const reqId = payload.requestId || 'r1';
        if (payload.type === 'RTO_REQUEST' && payload.action === 'detect') {
          const resp = { type:'RTO_RESPONSE', requestId:reqId, ok:true, data:{ ready:true, version:'7.10.1' } };
          listeners.forEach(cb => cb({ data: resp } as any));
        } else if (payload.type === 'open' || payload.type === 'navigate' || payload.type === 'focus') {
          const resp = { type:'RTO_RESPONSE', requestId:reqId, ok:true, data:{} };
          listeners.forEach(cb => cb({ data: resp } as any));
        } else if (payload.type === 'command' && payload.action === 'getHtml') {
          const resp = { type:'getHtmlResult', requestId:reqId, status:'ok', data:'<div class="tF2Cxc">OK</div>' };
          listeners.forEach(cb => cb({ data: resp } as any));
        }
      });
    }
  };
  return w;
}

describe('E2E smoke (mocked)', () => {
  let originalAdd:any, originalRemove:any, originalPost:any;
  let fake: any;

  beforeEach(() => {
    fake = mockWindow();
    // patch global window references used by SDK
    (globalThis as any).window = fake;
    (globalThis as any).addEventListener = fake.addEventListener;
    (globalThis as any).removeEventListener = fake.removeEventListener;
  });

  afterEach(() => {
    delete (globalThis as any).window;
  });

  it('detect → open → domType → getHtml', async () => {
    const rto = createRtoClient({ timeoutMs: 5000, target: (globalThis as any).window });
    const det = await rto.detect();
    expect(det?.ready).toBe(true);
    await rto.open({ url: 'https://www.google.com/', tabKey: 'google', focus: true, newTab: true, singleton: true });
    // domType is fire-and-forget in our mock (we just return ok)
    await rto.domType({ tabKey: 'google', selector: 'input[name="q"]', text: 'RTO' });
    const html = await rto.getHtml({ tabKey: 'google', selector: '#search .tF2Cxc', property: 'outerHTML' });
    expect(html).toContain('tF2Cxc');
  });
});
